const util = require('../src/util');
const { assert } = require('chai');

describe('Lib: Util', function() {

  describe('get32IntFromBuffer', function() {

    it('should extract even-long-hex number from buffer smaller than 4 bytes', function() {
      const hex = '7b';
      const buffer = Buffer.from(hex, 'hex');
      assert.equal(util.get32IntFromBuffer(buffer), parseInt(hex, 16));
    });

    it('should extract odd-long-hex number from buffer smaller than 4 bytes', function() {
      const hex = '86a';
      const buffer = Buffer.from('0' + hex, 'hex');
      assert.equal(util.get32IntFromBuffer(buffer), parseInt(hex, 16));
    });

    it('should extract zero', function() {
      const hex = '0';
      const buffer = Buffer.from('0' + hex, 'hex');
      assert.equal(util.get32IntFromBuffer(buffer), parseInt(hex, 16));
    });

    it('should return NaN from an out of range offset', function() {
      const hex = '0';
      const buffer = Buffer.from('0' + hex, 'hex');
      assert.isNaN(util.get32IntFromBuffer(buffer, 12));
    });

    it('should extract even-long-hex number from buffer bigger than 4 bytes using offset', function() {
      const hex = '7b';
      const buffer = Buffer.from('22000000' + hex + '1111', 'hex');
      assert.equal(util.get32IntFromBuffer(buffer, 1), parseInt(hex, 16));
    });

    it('should extract odd-long-hex number from buffer bigger than 4 bytes using offset', function() {
      const hex = '7ba';
      const buffer = Buffer.from('222200000' + hex + '1111', 'hex');
      assert.equal(util.get32IntFromBuffer(buffer, 2), parseInt(hex, 16));
    });

  });

  describe('hasPublicComponents', function() {

    const components = { n: '', e: '' };

    it('should return true for object with all required components', function() {
      assert.isTrue(util.hasPublicComponents(components));
    });

    it('should return false if some components are missing ', function() {
      assert.isFalse(util.hasPublicComponents({ ...components, e: undefined }));
    });

  });

  describe('hasPrivateComponents', function() {

    const components = { n: '', e: '', d: '', p: '', q: '', dp: '', dq: '', qi: '' };

    it('should return true for object with all required components', function() {
      assert.isTrue(util.hasPrivateComponents(components));
    });

    it('should return false if some components are missing ', function() {
      assert.isFalse(util.hasPrivateComponents({ ...components, qi: undefined }));
    });

  });

});