const { assert } = require('chai');
const NodeRSA = require('node-rsa');
const key = require('../src/key');
const RSAKeyError = require('../src/error');
const { PRIVATE, PUBLIC, PKCS1, PKCS8, PEM, DER, BASE64 } = require('../src/constants');

describe('Lib: key', function() {

  describe('decode', function() {

    let testKey;

    before(function() {
      testKey = new NodeRSA({ b: 256 });
    });

    describe('DER keys', function() {

      it('should decode private PKCS8 DER key', function() {
        const testEncoded = testKey.exportKey('pkcs8-private-der');
        const decoded = key.decode(testEncoded);
        assert.isObject(decoded);
      });

      it('should decode public PKCS8 DER key', function() {
        const testEncoded = testKey.exportKey('pkcs8-public-der');
        const decoded = key.decode(testEncoded);
        assert.isObject(decoded);
      });

      it('should decode private PKCS1 DER key', function() {
        const testEncoded = testKey.exportKey('pkcs1-private-der');
        const decoded = key.decode(testEncoded);
        assert.isObject(decoded);
      });

      it('should decode public PKCS1 DER key', function() {
        const testEncoded = testKey.exportKey('pkcs1-public-der');
        const decoded = key.decode(testEncoded);
        assert.isObject(decoded);
      });

      it('should decode DER key provided in a string', function() {
        const testEncoded = testKey.exportKey('pkcs1-public-der').toString(BASE64);
        const decoded = key.decode(testEncoded);
        assert.isObject(decoded);
      });

    });

    describe('PEM keys', function() {

      it('should decode private PKCS8 PEM key', function() {
        const testEncoded = testKey.exportKey('pkcs8-private-pem');
        const decoded = key.decode(testEncoded);
        assert.isObject(decoded);
      });

      it('should decode public PKCS8 PEM key', function() {
        const testEncoded = testKey.exportKey('pkcs8-public-pem');
        const decoded = key.decode(testEncoded);
        assert.isObject(decoded);
      });

      it('should decode private PKCS1 PEM key', function() {
        const testEncoded = testKey.exportKey('pkcs1-private-pem');
        const decoded = key.decode(testEncoded);
        assert.isObject(decoded);
      });

      it('should decode public PKCS1 PEM key', function() {
        const testEncoded = testKey.exportKey('pkcs1-public-pem');
        const decoded = key.decode(testEncoded);
        assert.isObject(decoded);
      });

    });

    describe('Wrong keys', function() {

      it('should fail with a non-key string provided', function() {
        assert.throws(() => {
          key.decode('clearly not a key');
        }, RSAKeyError);
      });

      it('should fail with a non-key buffer provided', function() {
        assert.throws(() => {
          key.decode(Buffer.from('clearly not a key', 'utf8'));
        }, RSAKeyError);
      });

    });

  });

  describe('encode', function() {

    describe('DER keys', function() {

      let testKey, components;

      before(function() {
        testKey = new NodeRSA({ b: 256 });
        components = key.decode(testKey.exportKey('pkcs8-private-der'));
      });

      it('should encode private PKCS8 DER key', function() {
        const testEncoded = testKey.exportKey('pkcs8-private-der');
        const encoded = key.encode(components, DER, PRIVATE, PKCS8);
        assert.equal(encoded.toString(BASE64), testEncoded.toString(BASE64));
      });

      it('should encode public PKCS8 DER key', function() {
        const testEncoded = testKey.exportKey('pkcs8-public-der');
        const encoded = key.encode(components, DER, PUBLIC, PKCS8);
        assert.equal(encoded.toString(BASE64), testEncoded.toString(BASE64));
      });

      it('should encode private PKCS1 DER key', function() {
        const testEncoded = testKey.exportKey('pkcs1-private-der');
        const encoded = key.encode(components, DER, PRIVATE, PKCS1);
        assert.equal(encoded.toString(BASE64), testEncoded.toString(BASE64));
      });

      it('should encode public PKCS1 DER key', function() {
        const testEncoded = testKey.exportKey('pkcs1-public-der');
        const encoded = key.encode(components, DER, PUBLIC, PKCS1);
        assert.equal(encoded.toString(BASE64), testEncoded.toString(BASE64));
      });

    });

    describe('PEM keys', function() {

      let testKey, components;

      before(function() {
        testKey = new NodeRSA({ b: 256 });
        components = key.decode(testKey.exportKey('pkcs8-private-pem'));
      });

      it('should encode private PKCS8 PEM key', function() {
        const testEncoded = testKey.exportKey('pkcs8-private-pem');
        const encoded = key.encode(components, PEM, PRIVATE, PKCS8);
        assert.equal(encoded.toString(BASE64), testEncoded.toString(BASE64));
      });

      it('should encode public PKCS8 PEM key', function() {
        const testEncoded = testKey.exportKey('pkcs8-public-pem');
        const encoded = key.encode(components, PEM, PUBLIC, PKCS8);
        assert.equal(encoded.toString(BASE64), testEncoded.toString(BASE64));
      });

      it('should encode private PKCS1 PEM key', function() {
        const testEncoded = testKey.exportKey('pkcs1-private-pem');
        const encoded = key.encode(components, PEM, PRIVATE, PKCS1);
        assert.equal(encoded.toString(BASE64), testEncoded.toString(BASE64));
      });

      it('should encode public PKCS1 PEM key', function() {
        const testEncoded = testKey.exportKey('pkcs1-public-pem');
        const encoded = key.encode(components, PEM, PUBLIC, PKCS1);
        assert.equal(encoded.toString(BASE64), testEncoded.toString(BASE64));
      });

    });

    describe('Unknown format', function() {

      let testKey, components;

      before(function() {
        testKey = new NodeRSA({ b: 256 });
        components = key.decode(testKey.exportKey('pkcs8-private-pem'));
      });

      it('should fail when unknown format is specified', function() {
        assert.throws(() => {
          key.encode(components, 'clearly unknown format', PRIVATE, PKCS8);
        }, RSAKeyError);
      });

    });

  });

});