const { assert } = require('chai');
const { BASE64, PRIVATE, PUBLIC, PKCS1, PKCS8 } = require('../src/constants');
const NodeRSA = require('node-rsa');
const der = require('../src/formats/der');
const { get32IntFromBuffer } = require('../src/util');
const RSAKeyError = require('../src/error');

describe('Format: DER', function() {

  describe('isDER', function() {

    let testKey;

    before(function() {
      testKey = new NodeRSA({ b: 256 });
    });

    it('should recognize public PKCS8 DER key', function() {
      assert.isTrue(der.isDER(testKey.exportKey('pkcs8-public-der')));
    });

    it('should recognize public PKCS1 DER key', function() {
      assert.isTrue(der.isDER(testKey.exportKey('pkcs1-public-der')));
    });

    it('should recognize private PKCS8 DER key', function() {
      assert.isTrue(der.isDER(testKey.exportKey('pkcs8-private-der')));
    });

    it('should recognize private PKCS1 DER key', function() {
      assert.isTrue(der.isDER(testKey.exportKey('pkcs1-private-der')));
    });

    it('should not recognize a PEM key', function() {
      assert.isFalse(der.isDER(testKey.exportKey('pkcs1-private-pem')));
    });

  });

  describe('decodeDER', function() {

    describe('with specified type and syntax', function() {

      let testKey, testComponents;

      before(function() {
        testKey = new NodeRSA({ b: 256 });
        testComponents = testKey.exportKey('components');
      });

      it('should decode private PKCS8 DER key', function() {
        const testEncoded = testKey.exportKey('pkcs8-private-der');
        const decoded = der.decodeDER(testEncoded, PRIVATE, PKCS8);
        assert.isObject(decoded, 'should return an object');
        assert.equal(decoded.n, testComponents.n.toString(BASE64), 'decoded modulus should match original');
        assert.equal(get32IntFromBuffer(Buffer.from(decoded.e, BASE64)), testComponents.e,
          'decoded public exponent should match original');
        assert.equal(decoded.d, testComponents.d.toString(BASE64), 'decoded private exponent should match');
        assert.equal(decoded.p, testComponents.p.toString(BASE64), 'decoded prime1 should match');
        assert.equal(decoded.q, testComponents.q.toString(BASE64), 'decoded prime2 should match');
        assert.equal(decoded.dp, testComponents.dmp1.toString(BASE64), 'decoded exponent1 should match');
        assert.equal(decoded.dq, testComponents.dmq1.toString(BASE64), 'decoded exponent2 should match');
        assert.equal(decoded.qi, testComponents.coeff.toString(BASE64), 'decoded coefficient should match');
      });

      it('should decode public PKCS8 DER key', function() {
        const testEncoded = testKey.exportKey('pkcs8-public-der');
        const decoded = der.decodeDER(testEncoded, PUBLIC, PKCS8);
        assert.isObject(decoded, 'should return an object');
        assert.equal(decoded.n, testComponents.n.toString(BASE64), 'decoded modulus should match original');
        assert.equal(get32IntFromBuffer(Buffer.from(decoded.e, BASE64)), testComponents.e,
          'decoded public exponent should match original');
      });

      it('should decode private PKCS1 DER key', function() {
        const testEncoded = testKey.exportKey('pkcs1-private-der');
        const decoded = der.decodeDER(testEncoded, PRIVATE, PKCS1);
        assert.isObject(decoded, 'should return an object');
        assert.equal(decoded.n, testComponents.n.toString(BASE64), 'decoded modulus should match original');
        assert.equal(get32IntFromBuffer(Buffer.from(decoded.e, BASE64)), testComponents.e,
          'decoded public exponent should match original');
        assert.equal(decoded.d, testComponents.d.toString(BASE64), 'decoded private exponent should match');
        assert.equal(decoded.p, testComponents.p.toString(BASE64), 'decoded prime1 should match');
        assert.equal(decoded.q, testComponents.q.toString(BASE64), 'decoded prime2 should match');
        assert.equal(decoded.dp, testComponents.dmp1.toString(BASE64), 'decoded exponent1 should match');
        assert.equal(decoded.dq, testComponents.dmq1.toString(BASE64), 'decoded exponent2 should match');
        assert.equal(decoded.qi, testComponents.coeff.toString(BASE64), 'decoded coefficient should match');
      });

      it('should decode public PKCS1 DER key', function() {
        const testEncoded = testKey.exportKey('pkcs1-public-der');
        const decoded = der.decodeDER(testEncoded, PUBLIC, PKCS1);
        assert.isObject(decoded, 'should return an object');
        assert.equal(decoded.n, testComponents.n.toString(BASE64), 'decoded modulus should match original');
        assert.equal(get32IntFromBuffer(Buffer.from(decoded.e, BASE64)), testComponents.e,
          'decoded public exponent should match original');
      });

      it('should fail if wrong key type was provided', function() {
        assert.throws(() => {
          const testEncoded = testKey.exportKey('pkcs1-public-der');
          der.decodeDER(testEncoded, PRIVATE, PKCS1);
        }, RSAKeyError);
      });

      it('should fail if wrong key syntax was provided', function() {
        assert.throws(() => {
          const testEncoded = testKey.exportKey('pkcs1-private-der');
          der.decodeDER(testEncoded, PRIVATE, PKCS8);
        }, RSAKeyError);
      });

      it('should work with key provided in a string', function() {
        const testEncoded = testKey.exportKey('pkcs1-public-der').toString(BASE64);
        const decoded = der.decodeDER(testEncoded, PUBLIC, PKCS1);
        assert.isObject(decoded, 'should return an object');
        assert.equal(decoded.n, testComponents.n.toString(BASE64), 'decoded modulus should match original');
        assert.equal(get32IntFromBuffer(Buffer.from(decoded.e, BASE64)), testComponents.e,
          'decoded public exponent should match original');
      });

    });

    describe('without specifying type or syntax', function() {

      let testKey, testComponents;

      before(function() {
        testKey = new NodeRSA({ b: 256 });
        testComponents = testKey.exportKey('components');
      });

      it('should decode private PKCS8 DER key', function() {
        const testEncoded = testKey.exportKey('pkcs8-private-der');
        const decoded = der.decodeDER(testEncoded);
        assert.isObject(decoded, 'should return an object');
        assert.equal(decoded.n, testComponents.n.toString(BASE64), 'decoded modulus should match original');
        assert.equal(get32IntFromBuffer(Buffer.from(decoded.e, BASE64)), testComponents.e,
          'decoded public exponent should match original');
        assert.equal(decoded.d, testComponents.d.toString(BASE64), 'decoded private exponent should match');
        assert.equal(decoded.p, testComponents.p.toString(BASE64), 'decoded prime1 should match');
        assert.equal(decoded.q, testComponents.q.toString(BASE64), 'decoded prime2 should match');
        assert.equal(decoded.dp, testComponents.dmp1.toString(BASE64), 'decoded exponent1 should match');
        assert.equal(decoded.dq, testComponents.dmq1.toString(BASE64), 'decoded exponent2 should match');
        assert.equal(decoded.qi, testComponents.coeff.toString(BASE64), 'decoded coefficient should match');
      });

      it('should decode public PKCS8 DER key', function() {
        const testEncoded = testKey.exportKey('pkcs8-public-der');
        const decoded = der.decodeDER(testEncoded);
        assert.isObject(decoded, 'should return an object');
        assert.equal(decoded.n, testComponents.n.toString(BASE64), 'decoded modulus should match original');
        assert.equal(get32IntFromBuffer(Buffer.from(decoded.e, BASE64)), testComponents.e,
          'decoded public exponent should match original');
      });

      it('should decode private PKCS1 DER key', function() {
        const testEncoded = testKey.exportKey('pkcs1-private-der');
        const decoded = der.decodeDER(testEncoded);
        assert.isObject(decoded, 'should return an object');
        assert.equal(decoded.n, testComponents.n.toString(BASE64), 'decoded modulus should match original');
        assert.equal(get32IntFromBuffer(Buffer.from(decoded.e, BASE64)), testComponents.e,
          'decoded public exponent should match original');
        assert.equal(decoded.d, testComponents.d.toString(BASE64), 'decoded private exponent should match');
        assert.equal(decoded.p, testComponents.p.toString(BASE64), 'decoded prime1 should match');
        assert.equal(decoded.q, testComponents.q.toString(BASE64), 'decoded prime2 should match');
        assert.equal(decoded.dp, testComponents.dmp1.toString(BASE64), 'decoded exponent1 should match');
        assert.equal(decoded.dq, testComponents.dmq1.toString(BASE64), 'decoded exponent2 should match');
        assert.equal(decoded.qi, testComponents.coeff.toString(BASE64), 'decoded coefficient should match');
      });

      it('should decode public PKCS1 DER key', function() {
        const testEncoded = testKey.exportKey('pkcs1-public-der');
        const decoded = der.decodeDER(testEncoded);
        assert.isObject(decoded, 'should return an object');
        assert.equal(decoded.n, testComponents.n.toString(BASE64), 'decoded modulus should match original');
        assert.equal(get32IntFromBuffer(Buffer.from(decoded.e, BASE64)), testComponents.e,
          'decoded public exponent should match original');
      });

      it('should fail if non-DER key was provided', function() {
        assert.throws(() => {
          const testEncoded = testKey.exportKey('pkcs1-public-pem');
          der.decodeDER(testEncoded);
        }, RSAKeyError);
      });

    });

  });

  describe('encodeDER', function() {

    let testKey, testDecoded;

    before(function() {
      testKey = new NodeRSA({ b: 256 });
      testDecoded = der.decodeDER(testKey.exportKey('pkcs8-private-der'));
    });

    it('should encode private PKCS8 DER key', function() {
      const testEncoded = testKey.exportKey('pkcs8-private-der');
      const encoded = der.encodeDER(testDecoded, PRIVATE, PKCS8);
      assert.equal(encoded.toString('base64'), testEncoded.toString('base64'));
    });

    it('should encode public PKCS8 DER key', function() {
      const testEncoded = testKey.exportKey('pkcs8-public-der');
      const encoded = der.encodeDER(testDecoded, PUBLIC, PKCS8);
      assert.equal(encoded.toString('base64'), testEncoded.toString('base64'));
    });

    it('should encode private PKCS1 DER key', function() {
      const testEncoded = testKey.exportKey('pkcs1-private-der');
      const encoded = der.encodeDER(testDecoded, PRIVATE, PKCS1);
      assert.equal(encoded.toString('base64'), testEncoded.toString('base64'));
    });

    it('should encode public PKCS1 DER key', function() {
      const testEncoded = testKey.exportKey('pkcs1-public-der');
      const encoded = der.encodeDER(testDecoded, PUBLIC, PKCS1);
      assert.equal(encoded.toString('base64'), testEncoded.toString('base64'));
    });

    it('should fail if unknown type was specified', function() {
      assert.throws(() => {
        der.encodeDER(testDecoded, 'clearly-unknown-type', PKCS1);
      }, RSAKeyError);
    });

  });

});