/**
 * Credits: based on
 * https://github.com/rzcoder/node-rsa/blob/afac9f8e56824da72f038b1a2053837a49ceec05/src/formats/pkcs8.js
 */

const { Ber } = require('asn1');
const { BASE64 } = require('../constants');
const { get32IntFromBuffer, hasPrivateComponents, hasPublicComponents } = require('../util');
const RSAKeyError = require('../error');
const PUBLIC_RSA_OID = '1.2.840.113549.1.1.1';

/**
 * Decodes private PKCS8 DER key into components
 * @param {buffer|string} key
 * @param {string} [encoding=base64]
 * @return {{n, e, d, p, q, dp, dq, qi}}
 */
const decodePrivatePKCS8DER = (key, encoding = BASE64) => {
  try {

    const reader = new Ber.Reader(Buffer.from(key, encoding));
    reader.readSequence();
    reader.readInt(0);

    const header = new Ber.Reader(reader.readString(0x30, true));
    /* istanbul ignore if */
    if (header.readOID(0x06, true) !== PUBLIC_RSA_OID) {
      throw new RSAKeyError('Invalid key format');
    }

    const body = new Ber.Reader(reader.readString(0x04, true));
    body.readSequence();
    body.readString(2, true);

    return {
      n: body.readString(0x02, true).toString(BASE64),
      e: body.readString(0x02, true).toString(BASE64),
      d: body.readString(0x02, true).toString(BASE64),
      p: body.readString(0x02, true).toString(BASE64),
      q: body.readString(0x02, true).toString(BASE64),
      dp: body.readString(0x02, true).toString(BASE64),
      dq: body.readString(0x02, true).toString(BASE64),
      qi: body.readString(0x02, true).toString(BASE64)
    };
  }
  catch (err) {
    throw new RSAKeyError('Could not decode key', err);
  }
};

/**
 * Decodes public PKCS8 DER key into components
 * @param {buffer|string} key
 * @param {string} [encoding=base64]
 * @return {{n, e}}
 */
const decodePublicPKCS8DER = (key, encoding = BASE64) => {
  try {

    const reader = new Ber.Reader(Buffer.from(key, encoding));
    reader.readSequence();

    const header = new Ber.Reader(reader.readString(0x30, true));
    /* istanbul ignore if */
    if (header.readOID(0x06, true) !== PUBLIC_RSA_OID) {
      throw new RSAKeyError('Invalid key format');
    }

    const body = new Ber.Reader(reader.readString(0x03, true));
    body.readByte();
    body.readSequence();

    return {
      n: body.readString(0x02, true).toString(BASE64),
      e: body.readString(0x02, true).toString(BASE64)
    };
  }
  catch (err) {
    throw new RSAKeyError('Could not decode key', err);
  }

};

/**
 * Encodes private PKCS1 DER key from components
 * @param {object} components
 * @param {string} [encoding=base64]
 * @return {buffer}
 */
const encodePrivatePKCS8DER = (components, encoding = BASE64) => {

  if (hasPrivateComponents(components) === false) {
    throw new RSAKeyError('Could not encode private key, because some components are missing');
  }

  try {

    const nBuffer = Buffer.from(components.n, encoding);
    const dBuffer = Buffer.from(components.d, encoding);
    const pBuffer = Buffer.from(components.p, encoding);
    const qBuffer = Buffer.from(components.q, encoding);
    const dpBuffer = Buffer.from(components.dp, encoding);
    const dqBuffer = Buffer.from(components.dq, encoding);
    const qiBuffer = Buffer.from(components.qi, encoding);
    const eBuffer = Buffer.from(components.e, encoding);
    const eInt = get32IntFromBuffer(eBuffer);
    const length = nBuffer.length + dBuffer.length + pBuffer.length + qBuffer.length +
                   dpBuffer.length + dqBuffer.length + qiBuffer.length + 512;

    const body = new Ber.Writer({ size: length });
    body.startSequence();
    body.writeInt(0);
    body.writeBuffer(nBuffer, 2);
    body.writeInt(eInt);
    body.writeBuffer(dBuffer, 2);
    body.writeBuffer(pBuffer, 2);
    body.writeBuffer(qBuffer, 2);
    body.writeBuffer(dpBuffer, 2);
    body.writeBuffer(dqBuffer, 2);
    body.writeBuffer(qiBuffer, 2);
    body.endSequence();

    const writer = new Ber.Writer({ size: length });
    writer.startSequence();
    writer.writeInt(0);
    writer.startSequence();
    writer.writeOID(PUBLIC_RSA_OID);
    writer.writeNull();
    writer.endSequence();
    writer.writeBuffer(body.buffer, 4);
    writer.endSequence();

    return writer.buffer;
  }
  catch (err) {
    throw new RSAKeyError('Could not encode key', err);
  }

};

/**
 * Encodes public PKCS1 DER key from components
 * @param {object} components
 * @param {string} [encoding=base64]
 * @return {buffer}
 */
const encodePublicPKCS8DER = (components, encoding = BASE64) => {

  if (hasPublicComponents(components) === false) {
    throw new RSAKeyError('Could not encode private key, because some components are missing');
  }

  try {

    const nBuffer = Buffer.from(components.n, encoding);
    const eBuffer = Buffer.from(components.e, encoding);
    const eInt = get32IntFromBuffer(eBuffer);
    const length = nBuffer.length + 512;

    const body = new Ber.Writer({ size: length });
    body.writeByte(0);
    body.startSequence();
    body.writeBuffer(nBuffer, 2);
    body.writeInt(eInt);
    body.endSequence();

    const writer = new Ber.Writer({ size: length });
    writer.startSequence();
    writer.startSequence();
    writer.writeOID(PUBLIC_RSA_OID);
    writer.writeNull();
    writer.endSequence();
    writer.writeBuffer(body.buffer, 3);
    writer.endSequence();

    return writer.buffer;
  }
  catch (err) {
    throw new RSAKeyError('Could not encode key', err);
  }

};

module.exports = { decodePrivatePKCS8DER, decodePublicPKCS8DER, encodePrivatePKCS8DER, encodePublicPKCS8DER };