/**
 * Credits: based on
 * https://github.com/rzcoder/node-rsa/blob/afac9f8e56824da72f038b1a2053837a49ceec05/src/formats/pkcs1.js
 */

const { Ber } = require('asn1');
const { BASE64 } = require('../constants');
const { get32IntFromBuffer, hasPublicComponents, hasPrivateComponents } = require('../util');
const RSAKeyError = require('../error');

/**
 * Decodes private PKCS1 DER key into components
 * @param {buffer|string} key
 * @param {string} [encoding=base64]
 * @return {{n, e, d, p, q, dp, dq, qi}}
 */
const decodePrivatePKCS1DER = (key, encoding = BASE64) => {
  try {

    const reader = new Ber.Reader(Buffer.from(key, encoding));
    reader.readSequence();
    reader.readString(2, true);

    return {
      n: reader.readString(0x02, true).toString(BASE64),
      e: reader.readString(0x02, true).toString(BASE64),
      d: reader.readString(0x02, true).toString(BASE64),
      p: reader.readString(0x02, true).toString(BASE64),
      q: reader.readString(0x02, true).toString(BASE64),
      dp: reader.readString(0x02, true).toString(BASE64),
      dq: reader.readString(0x02, true).toString(BASE64),
      qi: reader.readString(0x02, true).toString(BASE64)
    };
  }
  catch (err) {
    throw new RSAKeyError('Could not decode key', err);
  }
};

/**
 * Decodes public PKCS1 DER key into components
 * @param {buffer|string} key
 * @param {string} [encoding=base64]
 * @return {{n, e}}
 */
const decodePublicPKCS1DER = (key, encoding = BASE64) => {
  try {

    const reader = new Ber.Reader(Buffer.from(key, encoding));
    reader.readSequence();

    const result = {
      n: reader.readString(0x02, true),
      e: reader.readString(0x02, true).toString(BASE64)
    };

    /* If a private key was used */
    if (get32IntFromBuffer(result.n) === 0) {
      result.n = result.e;
      result.e = reader.readString(0x02, true).toString(BASE64);
    }
    else {
      result.n = result.n.toString(BASE64);
    }

    return result;
  }
  catch (err) {
    throw new RSAKeyError('Could not decode key', err);
  }
};

/**
 * Encodes private PKCS1 DER key from components
 * @param {object} components
 * @param {string} [encoding=base64]
 * @return {buffer}
 */
const encodePrivatePKCS1DER = (components, encoding = BASE64) => {

  if (hasPrivateComponents(components) === false) {
    throw new RSAKeyError('Could not encode private key, because some components are missing');
  }

  try {
    const nBuffer = Buffer.from(components.n, encoding);
    const dBuffer = Buffer.from(components.d, encoding);
    const pBuffer = Buffer.from(components.p, encoding);
    const qBuffer = Buffer.from(components.q, encoding);
    const dpBuffer = Buffer.from(components.dp, encoding);
    const dqBuffer = Buffer.from(components.dq, encoding);
    const qiBuffer = Buffer.from(components.qi, encoding);
    const eBuffer = Buffer.from(components.e, encoding);
    const eInt = get32IntFromBuffer(eBuffer);
    const length = nBuffer.length + dBuffer.length + pBuffer.length + qBuffer.length +
                   dpBuffer.length + dqBuffer.length + qiBuffer.length + 512;

    const writer = new Ber.Writer({ size: length });
    writer.startSequence();
    writer.writeInt(0);
    writer.writeBuffer(nBuffer, 2);
    writer.writeInt(eInt);
    writer.writeBuffer(dBuffer, 2);
    writer.writeBuffer(pBuffer, 2);
    writer.writeBuffer(qBuffer, 2);
    writer.writeBuffer(dpBuffer, 2);
    writer.writeBuffer(dqBuffer, 2);
    writer.writeBuffer(qiBuffer, 2);
    writer.endSequence();
    return writer.buffer;
  }
  catch (err) {
    throw new RSAKeyError('Could not encode key', err);
  }
};

/**
 * Encodes public PKCS1 DER key from components
 * @param {object} components
 * @param {string} [encoding=base64]
 * @return {buffer}
 */
const encodePublicPKCS1DER = (components, encoding = BASE64) => {

  if (hasPublicComponents(components) === false) {
    throw new RSAKeyError('Could not encode private key, because some components are missing');
  }

  try {

    const nBuffer = Buffer.from(components.n, encoding);
    const eBuffer = Buffer.from(components.e, encoding);
    const eInt = get32IntFromBuffer(eBuffer);
    const length = nBuffer.length + 512;

    const writer = new Ber.Writer({ size: length });
    writer.startSequence();
    writer.writeBuffer(nBuffer, 2);
    writer.writeInt(eInt);
    writer.endSequence();
    return writer.buffer;
  }
  catch (err) {
    throw new RSAKeyError('Could not encode key', err);
  }
};

module.exports = { decodePrivatePKCS1DER, decodePublicPKCS1DER, encodePrivatePKCS1DER, encodePublicPKCS1DER };