const der = require('./der');
const { BASE64, PUBLIC, PRIVATE, PKCS1, PKCS8, UTF8 } = require('../constants');
const RSAKeyError = require('../error');
const { bufferStringify } = require('../util');

/**
 * Checks if provided key is in PEM format and returns format version
 * @param {string} key
 * @return {object|null}
 */
const testPEM = key => {

  const keyString = bufferStringify(key, UTF8);

  const rxPEM = /^-----BEGIN (PRIVATE|PUBLIC|RSA PUBLIC|RSA PRIVATE) KEY-----\r?\n([a-zA-Z0-9\r\n+\\/=]*)\r?\n-----END \1 KEY-----/m;
  const match = keyString.match(rxPEM);

  return match === null
    ? null
    : {
      syntax: match[1].includes('RSA') ? PKCS1 : PKCS8,
      type: match[1].includes('PRIVATE') ? PRIVATE : PUBLIC,
      der: match[2].replace(/[\r\n ]/g, '')
    };

};

/**
 * Decodes PEM key into components
 * @param {string} key
 * @param {object} [info]
 * @return {object}
 */
const decodePEM = (key, info) => {

  const keyString = bufferStringify(key, UTF8);

  if (typeof info === 'undefined') {
    info = testPEM(keyString);
  }
  if (info === null) {
    throw new RSAKeyError('This is not a PEM key');
  }

  return der.decodeDER(info.der, info.type, info.syntax);
};

/**
 * Encodes PEM key from components
 * @param {object} components
 * @param {string} type
 * @param {string} syntax
 * @return {string}
 */
const encodePEM = (components, type, syntax) => {

  let insert;
  switch (syntax) {
    case PKCS1:
      insert = 'RSA ';
      break;
    case PKCS8:
      insert = '';
      break;
    default:
      throw new RSAKeyError(`Unrecognized key syntax: ${syntax}.`);
  }

  const keyString = der.encodeDER(components, type, syntax).toString(BASE64);
  const keyLines = keyString.match(/.{1,64}/g).join('\n');

  return `-----BEGIN ${insert}${type.toUpperCase()} KEY-----\n${keyLines}\n-----END ${insert}${type.toUpperCase()} KEY-----`;

};

module.exports = { testPEM, decodePEM, encodePEM };