const { BASE64 } = require('../constants');
const pkcs1 = require('./pkcs1');
const pkcs8 = require('./pkcs8');
const RSAKeyError = require('../error');
const { bufferStringify } = require('../util');

/**
 * Returns DER function to encode/decode
 * @private
 * @param {string} type
 * @param {string} syntax
 * @param {string} prefix
 * @return {function}
 */
function getDERFunction(type, syntax, prefix) {
  const name = prefix + type.slice(0, 1).toUpperCase() + type.slice(1).toLowerCase() + syntax.toUpperCase() + 'DER';
  const func = { pkcs1, pkcs8 }[syntax.toLowerCase()][name];
  if (typeof func !== 'function') {
    throw new RSAKeyError(`Unknown key type ${type} or syntax ${syntax} specified for DER encoding/decoding`);
  }
  return func;
}

/**
 * Checks if provided key is in DER format
 * @param {buffer|string} key
 * @return {boolean}
 */
const isDER = key => {
  const keyString = bufferStringify(key, BASE64);
  const rxDER = /^[a-zA-Z0-9\r\n+\\/=]*$/;
  return keyString.match(rxDER) !== null;
};

/**
 * Decodes DER key into components
 * @param {buffer|string} key
 * @param {string} [type]
 * @param {string} [syntax]
 * @return {object}
 */
const decodeDER = (key, type, syntax) => {

  if (typeof type === 'string' && typeof syntax === 'string') {
    return getDERFunction(type, syntax, 'decode')(key);
  }
  else {

    try { return pkcs1.decodePrivatePKCS1DER(key); }
    catch (err) { /* key did not match function type */ }

    try { return pkcs8.decodePrivatePKCS8DER(key); }
    catch (err) { /* key did not match function type */ }

    try { return pkcs1.decodePublicPKCS1DER(key); }
    catch (err) { /* key did not match function type */ }

    try { return pkcs8.decodePublicPKCS8DER(key); }
    catch (err) { /* key did not match function type */ }

    throw new RSAKeyError('Could not decode DER key using any known format');

  }
};

/**
 * Encodes DER key from components
 * @param {object} components
 * @param {string} type
 * @param {string} syntax
 * @return {buffer}
 */
const encodeDER = (components, type, syntax) => {
  return getDERFunction(type, syntax, 'encode')(components);
};

module.exports = { isDER, decodeDER, encodeDER };