# rsa-key

[![Build Status](https://travis-ci.org/racbart/rsa-key.svg?branch=master)](https://travis-ci.org/racbart/rsa-key)
[![Known Vulnerabilities](https://snyk.io/test/github/racbart/rsa-key/badge.svg)](https://snyk.io/test/github/racbart/rsa-key)
[![Coverage Status](https://coveralls.io/repos/github/racbart/rsa-key/badge.svg?branch=master)](https://coveralls.io/github/racbart/rsa-key?branch=master)

Node.js library to convert between RSA key formats and get information about the key. No OpenSSL needed. Detecting input format type, so you don't need to specify or even know the format in which your RSA key is stored.

## Quickstart

Installation:

```bash
npm install rsa-key
```

Usage examples:

```javascript
var RSAKey = require('rsa-key');
var keyInAnyFormat = /* your RSA key in any supported format, in string or buffer */;

/* Import the key */
var key = new RSAKey(keyInAnyFormat);

/* Export key in default format (PEM PKCS8) */
var output = key.exportKey();

/* Export key in a specified format; see details in usage docs */
var output = key.exportKey('public', 'pem', 'pkcs1');
var output = key.exportKey('der');

/* Get key fingerprint */
var fingerprint = key.getFingerprint();
```

### Usage in old versions of Node (4 and 6)
You should use ES5 version of the library which is in a separate NPM package:

```javascript
var RSAKey = require('rsa-key-es5');
```

## Supported formats

Currently supported formats are PEM, DER. Supported syntaxes are PKCS1 and PKCS8. I might add JWK at some time in the future.

## Usage

### Creating instance

```javacript
var RSAKey = require('rsa-key');
var emptyKey = new RSAKey();
```

You can provide key in any supported format as constructor argument (same as in `importKey` below).

```javacript
var RSAKey = require('rsa-key');
var key = new RSAKey(keyData);
```

### Importing and exporting keys

#### `key.importKey(keyData)`

Imports key into the object. It's equivalent to creating the object providing `keyData` in constructor argument.

`keyData` can be in any of the supported formats. It can be a string or a buffer or another RSAKey object. If buffer is provided, it's converted to a string with `utf8` encoding when testing for PEM key and `base64` encoding when testing for DER key.

```javacript
key.importKey(keyData);
```

You can call `importKey` multiple times to reuse the object. Each call wipes the object and replaces with new key data.

#### `key.exportKey([param, ..., param])`

Exports key into the format specified by params provided as arguments. Arguments are strings specifying key format, syntax and key type (private or public). Valid values are:

* `pem`, `der` to specify output key format, default: `pem`
* `pkcs1`, `pkcs8` to specify output syntax for PEM and DER keys, default: `pkcs8`
* `private`, `public` to specify key type to export (default: actual stored key type)

Params can be specified in any order:

```javascript
// These are equivalent
var output = key.exportKey('der', 'pkcs1', 'public');
var output = key.exportKey('pkcs1', 'public', 'der');
```

If conflicting params are provided, the latter overwrites the earlier:

```javascript
var output = key.exportKey('private', 'pem', 'pkcs8', 'der'); // exports in DER format (PEM is overwritten)
```

You can skip any params to use defaults:

```javascript
var output = key.exportKey('der');  // 'pkcs8' is applied as default
var output = key.exportKey();       // 'pem' and 'pkcs8' are applied as defaults
```

If key object stores a private key, the private key will be exported if no key type params were specified (`public` or `private`). If key object stores a public key, the public key will be exported. If you specify `private` param for an object storing only the public key, an error will be thrown.

```javascript
var privKey = new RSAKey(privateKeyData);   // importing private key
var output = privKey.exportKey();           // private key is exported
var output = privKey.exportKey('public');   // public key is exported

var pubKey = new RSAKey(publicKeyData);     // importing public key
var output = pubKey.exportKey();            // public key is exported
var output = pubKey.exportKey('private');   // error is thrown
```
If you know you want to export a public key, it's recommended to always specify `public` param, to avoid mistakes (when you think the object has only public key stored, but it mistakenly has the private key). Calling without key type params is useful if you want to convert the key to desired format regardless of key type.

Returned value type is:

* *string* for PEM export
* *Buffer* object for DER export

### Getting key information

#### `key.getFingerprint()`

Returns string with key fingerprint. It's always the same for the same key, regardless of input format which you used when importing the key (fingerprint is made on the public key in DER format using PKCS8 syntax, just as OpenSSL's standard fingerprint).

```javascript
var fingerprint = key.getFingerprint();
```

#### `key.hasKey()` and `key.isEmpty()`

Checks if key is loaded into the object. Returns boolean value.

```javascript
var key = new RSAKey();         // no key imported
var isEmpty = key.isEmpty();    // true
var hasKey = key.hasKey();      // false
```

#### `key.getType()`

Returns string `private` or `public`, depending on which key was imported. For empty key returns *null*.

#### `key.isPrivate()`

Checks if key object has private key stored. Returns boolean value. For empty key returns *false*.

#### `key.isPublic([strict = false])`

Checks if key has public key stored. Returns boolean value. For empty key returns *false*. For private key returns *true* (because it can be converted to public key), unless optional argument is set to *true*.

```javascript
var key = new RSAKey(privateKey);           // importing private key
var hasPublicKey = key.isPublic();          // true
var isExactlyPublic = key.isPublic(true);   // false
```

#### `key.getKeySize()`

Returns key size in bits. For empty key, returns *0*.

## License

MIT License